/***************************************************************
 * Name:      main.cpp
 * Purpose:   Illustrate the usage of the MPEG-7 Feature Extraction library
 *            uses OpenCV for image handling (as does the library)
 * Author:    Muhammet Bastan (bastan@cs.bilkent.edu.tr)
 * Created:   2010-02-20
 * Copyright: Muhammet Bastan (http://www.cs.bilkent.edu.tr/~bastan)
 * License:
 **************************************************************/

#include <iostream>
#include "FexWrite.h"

#include "highgui.h"
#include "cv.h"

int main( int argc, char* argv[] )
{
    IplImage* image = 0;

    // load an image
    if( argc < 2 )
        image = cvLoadImage("motor.jpg");
    else
        image = cvLoadImage( argv[1] );

    // display the image, if OpenCV is compiled with GTK or Carbon support!
    //cvNamedWindow("image");
    //cvShowImage("image", image);
    //cvWaitKey();

    ///-- extract global image descriptors --

    // create a Frame object (see include/Frame.h)
    // allocate memory for 3-channel color and 1-channel gray image and mask
    Frame* frame = new Frame( image->width, image->height, true, true, true);

    // set the image of the frame
    frame->setImage(image);

    // compute and display the descriptors for the 'image'
    // CSD of size 32
    FexWrite::computeWriteCSD(frame, 32);

    // SCD of size 128
    FexWrite::computeWriteSCD( frame, true, 128 );

    // CLD with numberOfYCoeff (28), numberOfCCoeff (15)
    FexWrite::computeWriteCLD( frame, 28, 15 );

    // DCD with normalization (to MPEG-7 ranges), without variance, without spatial coherency
    FexWrite::computeWriteDCD( frame, true, false, false );

    // EHD
    FexWrite::computeWriteEHD( frame );

    // HTD: we need to compute and set the grayscale image of the frame
    // create the grayscale image
    IplImage* gray = cvCreateImage( cvGetSize(image), 8, 1 );
    cvCvtColor( image, gray, CV_BGR2GRAY );
    frame->setGray( gray );
    cvReleaseImage(&gray);
    // full layer (both energy and deviation)
    FexWrite::computeWriteHTD( frame, 1 );

    // release the image
    cvReleaseImage(&image);

    // load another image
    image = cvLoadImage("timsah.jpg");
    //cvShowImage("image", image);
    //cvWaitKey();

    // set the image of 'frame'
    // we should resize the 'frame' first, the dimensions may be different
    // therefore, reallocation may be required
    frame->resize( image->width, image->height);

    // set the image
    frame->setImage(image);

    // here, we can compute the descriptors as above..
    // ..

    ///--- extract region-based descriptos ---

    // we need a foreground mask for the region
    // lets first create a dummy foreground mask to use as the region mask

    IplImage* mask = cvCreateImage( cvGetSize(image), 8, 1 );
    cvZero(mask);
    // draw a filled rectangle/circle, with foreground value 200
    int regVal = 200;   // (1...255)
    //cvRectangle( mask, cvPoint(60,20), cvPoint(190,230), cvScalar(regVal), CV_FILLED );
    //cvRectangle( mask, cvPoint(20,80), cvPoint(190,130), cvScalar(regVal), CV_FILLED );
    cvCircle( mask, cvPoint(120, 100), 50, cvScalar(regVal), -1);
    cvCircle( mask, cvPoint(120, 200), 60, cvScalar(regVal), -1);

    // display the mask
    //cvNamedWindow("mask");
    //cvShowImage("mask", mask);
    //cvWaitKey();

    // set the mask of 'frame', pixels having regVal are set to 255, background pixels to 0
    frame->setMaskAll( mask, regVal, 255, 0 );

    // SCD of the region
    FexWrite::computeWriteSCD( frame, false, 128 );

    // RSD (region shape)
    FexWrite::computeWriteRSD( frame );

    // CShD (contour shape)
    FexWrite::computeWriteCShD( frame );

    FexWrite::computeWriteCSD( frame, 64 );

    // at this point, if you want to extract descriptors
    // from the whole image, reset the mask
    frame->resetMaskAll();
    FexWrite::computeWriteCSD( frame, 64 );
    FexWrite::computeWriteEHD( frame );

    // ...
    // ...

    // release frame
    delete frame;

    // destroy the windows
    //cvDestroyWindow("image");
    //cvDestroyWindow("mask");

    return 0;
}
