%DISPERROR Display error matrix with information on classifiers and datasets
%
%	DISPERROR(DATA,CLASSF,ERROR,STD)
% 
% INPUT
%		DATA     Cell array of M datasets or dataset names (strings)
%		CLASSF   Cell array of N mappings or mapping names (strings)
%		ERROR    M*N matrix of (average) error estimates 
%		STD      M*N matrix of standard devations on ERROR (optional)
% OUTPUT
%
% DESCRIPTION
% Displays the matrix ERROR matrix with error estimates for N
% classifiers related to M datasets. This routine is called by TESTC
% and CROSVALL to display results.
%
% EXAMPLE
% testsets  = {gendath gendatb gendatd(100,5)}
% trainsets = {gendath gendatb gendatd(100,5)}
% classifiers = {nmc fisherc qdc svc}
% testc(testsets,map(trainsets,classifiers))
%
% SEE ALSO
% MAPPINGS, DATASETS, TESTC, CROSSVAL

% $Id: disperror.m,v 1.5 2003/10/19 14:43:00 bob Exp $

function disperror (data,classf,err,stdev)

	% Check arguments.
	if (nargin > 3) & (any(size(err) ~= size(stdev)))
		error('size of matrix with standard deviations should match matrix with errors')
	end
	if (~iscell(classf)) | (~isstr(classf{1}) & ~ismapping(classf{1}))
		error('cell array of mappings or mapping names expected')
	end
	if (~iscell(data)) | (~isstr(data{1}) & ~isdataset(data{1}))
		error('cell array of datasets or datasets names expected')
	end

	[m,n] = size(err);
	if (length(data) ~= m)
		error('size of dataset cell array should equal number of rows in error matrix');
	end
	if (length(classf) ~= n)
		error('size of classifier cell array should equal number of columns in error matrix');
	end

	% If datasets are supplied, extract their names.
	for j = 1:m
		if (isdataset(data{j}))
			data{j} = getname(data{j},20);
		end
	end

	% If classifiers are supplied, extract their names.
	for j = 1:n
		if (ismapping(classf{j}))
			classf{j} = getname(classf{j});
		end
	end

	if (n == 1)
		fprintf(' %s \n\n',classf{1});
	else
		fprintf('\n');
		for i = 1:n
			fprintf('\n  clsf_%i : %s',i,classf{i});
		end
		fprintf('\n\n                      ');
		for i = 1:n
			fprintf('  clsf_%i',i);
		end
		fprintf('\n\n');
	end

	for j = 1:m
		fprintf('  %s',data{j});
		fprintf(' %7.3f',err(j,:));
		if (nargin > 3)
			fprintf('\n                      ');
			fprintf(' %7.3f',stdev(j,:));
			fprintf('\n');
		end
		fprintf('\n');
	end
	fprintf('\n');
	
return
