%EMCLUST Expectation-Maximization clustering
%
%  [LABELS,W_EM] = EMCLUST (A,W_CLUST,K,LABTYPE,FID)
%
% INPUT
%   A         Dataset, possibly labeled
%   W_CLUST   Cluster model mapping, untrained (default: nmc)
%   K         Number of clusters (default: 2)
%   LABTYPE   Label type: 'crisp' or 'soft' (default: label type of A)
%   FID       File ID to write progress to (default [], see PRPROGRESS)
%
% OUTPUT
%   LABELS    Integer labels for the objects in A pointing to their cluster
%   W_EM      EM clustering mapping
%
% DESCRIPTION
% The untrained classifier mapping W_CLUST is used to update an initially
% labeled dataset A by iterating the following two steps:
%   1. Train W   :  W_EM = A*W_CLUST
%   2. Relabel A :  A    = dataset(A,labeld(A*W_EM*classc))
% This is repeated until the labeling does not change anymore. The final
% classification matrix is returned in B. The final crisp labeling is returned
% in LABELS. W_EM may be used for assigning new objects.
%
% If K is given, a random initialisation for K clusters is made and labels
% of A are neglected. 
%
% LABTYPE determines the type of labeling: 'crisp' or 'soft'. Default: label
% type of A. It is assumed W_CLUST can handle the LABTYPE requested.
%
% SEE ALSO
% MAPPINGS, DATASETS, KMEANS, PRPROGRESS

% Copyright: R.P.W. Duin, r.p.w.duin@prtools.org
% Faculty EWI, Delft University of Technology
% P.O. Box 5031, 2600 GA Delft, The Netherlands

% $Id: emclust.m,v 1.18 2004/06/20 14:26:07 duin Exp $

function [new_lab,w_em] = emclust (a,w_clust,n,type,fid)

	prtrace(mfilename);

	n_ini		= 500;			% Maximum size of subset to use for initialisation.
	epsilon = 1e-6;			% Stop when average labeling change drops below this.

	% Check arguments.
	if (nargin < 5), fid = []; end
	if (nargin < 4)
		prwarning(3,'No label type specified, using label type of dataset A.');
		type = []; 
	end
	if (nargin < 3) | isempty(n)
		prwarning(3,'No number of clusters specified, using number of classes in A.');
		n = []; 
	end
	if (nargin < 2) | isempty(w_clust)
		prwarning(2,'No clustering mapping specified, assuming NMC.');
		w_clust = nmc;   
	end

  isuntrained(w_clust);   % Assert that clustering mapping is untrained.

  % Determine number of clusters N and initialisation method.

	a = dataset(a); 
	islabtype(a,'crisp','soft');
	[m,k,c] = getsize(a); 
	rand_init = 1;
	if (isempty(n))
		if (c == 1)						% For one class, find two clusters.
			n = 2;
		else
			n = c;							
			rand_init = 0; 			% Use given classification as initialisation.
		end
	elseif n == c
		rand_init = 0;        % Use given classification as initialisation.
	end

	if (n < 1),  error('Number of clusters should be at least one.'); end
	if (n == 1), prwarning(4,'Clustering with 1 cluster is trivial.'); end

	% Set label type, if given.

	if ~isempty(type), a = setlabtype(a,type); end
	a = setprior(a,[]); % make sure that priors will be deleted
	
	% Initialise by performing KCENTRES on...

	if (rand_init)
		if (m > n_ini)       						% ... a random subset of A.
      prwarning(1,'Initializing by performing KCENTRES on a subset of %d samples.', n_ini);
			a_ini = +gendat(+a,n_ini);	
		else
      prwarning(1,'Initializing by performing KCENTRES on the training set.');
			a_ini = +a;								% ... the entire set A.
		end
		assign  = kcentres(+distm(a_ini),n);
		% Train initial classifier on labels generated by KCENTRES and find
		% initial hard labels. Use NMC instead of W_CLUST to make sure that we 
    % always have enough data to estimate the parameters.
		d = a*(dataset(a_ini,assign)*nmc);
  	if (islabtype(a,'soft'))
			new_lab = +d;
		else
			new_lab = d*labeld;
		end
	else                   
		new_lab = getlabels(a);		% Use given labeling.
	end

  % Ready for the work.
	prprogress(fid,'\nemclust optimization\n')
	iter = 0;
  if (islabtype(a,'soft'))                           
  	change = 1;
		a = setlabels(a,new_lab);
		lab = new_lab;
  	while (change > epsilon)       	% EM loop, run until labeling is stable.
  		w_em = a*w_clust;             % 1. Train classifier, density output.
  		b = a*(w_em*classc);          % 2. Assign probability to training samples.
  		a = settargets(a,b);          % 3. Insert probabilities as new labels.
  		change = mean(mean((+b-lab).^2)); lab = b;          
			prprogress(fid,'  change = %f (mse)\n',change);
			iter = iter+1;
			if iter > 500
				prwarning(1,'emclust stopped after 500 iterations')
				change = 0;
			end
  	end
	else  % crsip labels
  	lab = ones(m,1);
  	while (any(lab ~= new_lab))     % EM loop, run until labeling is stable.
  		a = setlabels(a,new_lab); 		% 0. Set labels and store old labels.
  		lab = new_lab;								% 
  		w_em = a*w_clust;             % 1. Compute classifier, crisp output.
  		b = a*w_em;               		% 2. Classify training samples.
  		new_lab = labeld(b);      		%	3. Insert classification as new labels.
			prprogress(fid,'  change = %d objects, no. of classes = %i\n', ...
				length(find(lab ~= new_lab)),length(unique(new_lab)));
			iter = iter+1;
			if iter > 500
				prwarning(1,'emclust stopped after 500 iterations')
				change = 0;
			end
  	end
  end
	prprogress(fid,'emclust finished\n')

return;
